' Ishido for CMM2
' Rev 1.0.0 William M Leue 31-Mar-2022

option default integer
option base 1

' Constants
const CSIZE       = 50
const STONE_CRAD  = 10

const BOARD_NCOLS  = 12
const BOARD_NROWS  = 8
const BOARD_WIDTH  = BOARD_NCOLS*CSIZE
const BOARD_HEIGHT = BOARD_NROWS*CSIZE
const BOARD_X      = 10
const BOARD_Y      = 75
const WITHIN       = 1
const BEYOND       = 2
const EMPTY        = 0

const NUM_COLORS  = 6
const NUM_SYMBOLS = 6
const NUM_STONES  = 2*NUM_SYMBOLS*NUM_COLORS
const NISTONES    = 6

const HEART   = 1
const STAR    = 2
const SQUARE  = 3
const CROSS   = 4
const TANGLE  = 5
const CRCLE   = 6

const BEYOND_COLOR = rgb(brown)
const WITHIN_COLOR = rgb(salmon)
const D1 = 100
const D2 = 200

const NUM_BONUSES = 12
const NUM_SCORE_PARTS = 6

' locations and sizes of info boxes to right of board
const TSPX = 650
const TSPW = 130
const TSY = 50
const TSH = 100
const SCY = 200
const SCH = 150
const PY = 400
const PH = 100

' options
const SHOW_MOVES_OPTION = 0
const SHOW_POUCH_OPTION = 0

' mouse channel
const MCHAN = 2

' keyboard commands
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const F1    = 145
const F12   = 156
const HOME  = 134
const ESC   = 27
const BKSPC = 8

const INVALID = -1

const STACK_SIZE = 100

const MAX_TOP_SCORES = 10
const TOPSCORE_W = 200
const TOPSCORE_H = 230
const TOPSCORE_X = mm.hres\2 - TOPSCORE_W\2
const TOPSCORE_Y = 100

const WIN_W = 600
const WIN_H = 400
const WIN_X = mm.hres\2 - WIN_W\2
const WIN_Y = mm.vres\2 - WIN_H\2

' Globals
dim mscores(4) = (1, 2, 4, 8)
dim pscores(3) = (1000, 500, 100)
dim score = 0
dim score_parts(NUM_SCORE_PARTS)
dim stones(NUM_STONES)
dim board(BOARD_WIDTH, BOARD_HEIGHT)
dim restart_board(BOARD_WIDTH, BOARD_HEIGHT)
dim pouch(NUM_STONES)
dim restart_pouch(NUM_STONES)
dim bonuses(NUM_BONUSES)
dim colors(NUM_COLORS) = (rgb(red), rgb(green), rgb(blue), rgb(yellow), rgb(cyan), rgb(magenta))
dim dark_colors(NUM_COLORS) = (rgb(D2,0,0), rgb(0,D2,0), rgb(0,0,D2), rgb(D1,D1,0), rgb(0,D1,D1), rgb(D1,0,D1))
dim nvertices(NUM_SYMBOLS) = (39, 11, 5, 13, 4, 0)
dim xv(39, NUM_SYMBOLS)
dim yv(39, NUM_SYMBOLS)
dim iblocs(2, NISTONES)
dim hasMouse = 0
dim num_pouch_stones = 0
dim num_board_stones = 0
dim n4ways = 0
dim running = 0

dim mouse_X = 0
dim mouse_Y = 0
dim left_click = 0

dim stack(3, STACK_SIZE)
dim sp = 0

dim topscore_names$(MAX_TOP_SCORES)
dim topscore_values(MAX_TOP_SCORES)

' Main Program
open "debug.txt" for output as #1
ReadIBLocs
ReadBonuses
ReadVertices
InitMouse
ShowStartScreen
NewGame
if hasMouse then
  GameLoopMouse
else
  GameLoopKeyboard
end if
end

' Initialize Mouse and Cursor if possible
sub InitMouse
  gui cursor on 1, 0, 0, rgb(yellow)
  on error skip 1
  controller mouse open MCHAN, LClick
  if mm.errno <> 0 then
    hasMouse = 0
  else
    hasMouse = 1
    settick 20, UpdateCursor
  end if
end sub

' Sync cursor and mouse
sub UpdateCursor
  gui cursor mouse(X), mouse(Y)
end sub

' Handle mouse clicks
sub LClick
  if left_click = 0 then
    mouse_X = mouse(X)
    mouse_Y = mouse(Y)
    left_click = 1
  end if
end sub

' Read the board coordinates for the initial stones
sub ReadIBLocs
  local i
  for i = 1 to NISTONES
    read iblocs(1, i)
    read iblocs(2, i)
  next i
end sub

' Read the 4-way play bonuses
sub ReadBonuses
  local i
  for i = 1 to NUM_BONUSES
    read bonuses(i)
  next i
end sub

' Read symbol vertices
sub ReadVertices
  local s, i
  for s = 1 to NUM_SYMBOLS
    if nvertices(s) > 0 then
      for i = 1 to nvertices(s)
        read xv(i, s)
        read yv(i, s)
      next i
    end if
  next s
end sub

' Init for a new game
sub NewGame
  local i
  cls
  ClearBoard
  FillPouch
  DoInitialDeal  
  sp = 0
  score = 0
  n4ways = 0
  running = 1
  for i = 1 to NUM_SCORE_PARTS
    score_parts(i) = 0
  next i
  DrawGame
end sub

' Clear the board
sub ClearBoard
  local row, col
  for row = 1 to BOARD_NROWS
    for col = 1 to BOARD_NCOLS
      board(col, row) = EMPTY
    next col
  next row
end sub

' Fill the pouch with stones and shuffle them
sub FillPouch
  local p, symbol, cx, n
  n = 0
  for symbol = 1 to NUM_SYMBOLS
    for cx = 1 to NUM_COLORS
      for p = 1 to 2
        inc n
        pouch(n) = EncodeStone(symbol, cx)
      next p
    next cx
  next symbol
  num_pouch_stones = NUM_STONES
  ShufflePouch
end sub  

' Shuffle stones in the pouch randomly
sub ShufflePouch
  local i, t, r
  for i = 1 to num_pouch_stones
    t = pouch(i)
    r = RandomIntegerInRange(1, NUM_STONES)
    pouch(i) = pouch(r)
    pouch(r) = t
  next i
end sub

' Deal the initial 6 stones to the board
' Assure symbol and color diversity
sub DoInitialDeal
  local us(NISTONES), uc(NISTONES), i, j, sx, cx, ok, row, col
  for i = 1 to NISTONES
    us(i) = 0 : uc(i) = 0
  next i
  for i = 1 to NISTONES
    do
      ok = 1
      sx = RandomIntegerInRange(1, NUM_SYMBOLS)
      for j = 1 to i-1
        if us(j) = sx then ok = 0      
      next j
    loop until ok
    us(i) = sx
    do
      ok = 1
      cx = RandomIntegerInRange(1, NUM_COLORS)
      for j = 1 to i-1
        if uc(j) = cx then ok = 0      
      next j
    loop until ok
    uc(i) = cx
    col = iblocs(1, i) : row = iblocs(2, i)
    board(col, row) = EncodeStone(sx, cx)
    RemoveStoneFromPouch sx, cx
  next i
  math add pouch(), 0, restart_pouch()
  math add board(), 0, restart_board()
end sub

' Draw a stone from the pouch and return it.
' This also updates the touchstone display.
sub DrawStoneFromPouch stone
  stone = 0
  if num_pouch_stones = 0 then
    exit sub
  else
    stone = pouch(num_pouch_stones)
    inc num_pouch_stones, -1
  end if
  DrawTouchstone
end sub

' Find a specific stone in the pouch and remove it
' (This is used only for the initial deal)
sub RemoveStoneFromPouch symbol, cx
  local stone = EncodeStone(symbol, cx)
  local i, j
  for i = 1 to num_pouch_stones
    if pouch(i) = stone then
      for j = i+1 to num_pouch_stones
        pouch(j-1) = pouch(j)
      next j
      exit for
    end if
  next i
  inc num_pouch_stones, -1
end sub

' Encode the stone value into an integer
function EncodeStone(symbol, cx)
  EncodeStone = symbol*100 + cx
end function

' Decode the stone values from the packed integer
sub DecodeStone stone, symbol, cx
  symbol = stone\100
  cx = stone - 100*symbol
end sub

' Draw the game, including the board, touchstone, score, and pouch
sub DrawGame
  gui cursor hide
  text mm.hres\2, 10, "Ishido - The Way of Stones", "CT", 4,, rgb(green)
  DrawBoard
  DrawTouchstone
  DrawScore
  DrawPouch
  gui cursor show
end sub

' Draw the playing board with current stones
sub DrawBoard
  local x, y, row, col, stone
  for row = 1 to BOARD_NROWS
    y = BOARD_Y + (row-1)*CSIZE 
    for col = 1 to BOARD_NCOLS
      x = BOARD_X + (col-1)*CSIZE
      stone = board(col, row)
      DrawBackground col, row, x, y
      if stone <> EMPTY then
        DrawStone x, y, stone
      end if
    next col
  next row
end sub

' Draw the touchstone
sub DrawTouchstone
  local x, y
  box TSPX, TSY, TSPW, TSH,, rgb(white), rgb(black)
  text TSPX+TSPW\2, TSY+5, "Touchstone", "CT",,, rgb(green)
  x = TSPX+TSPW\2 - CSIZE\2
  y = TSY+TSH\2 - CSIZE\2 + 10
  if num_pouch_stones > 0 then
    DrawStone x, y, pouch(num_pouch_stones)
  end if
end sub

' Draw the current score
sub DrawScore
  local x, y
  x = TSPX + TSPW\2
  y = SCY + SCH\2
  box TSPX, SCY, TSPW, SCH,, rgb(white), rgb(black)
  text x, SCY+5, "Score", "CT",,, rgb(green)
  text x, y, str$(score), "CT", 5
end sub

' Draw the pouch
sub DrawPouch
  local m$ = str$(num_pouch_stones) + " Stones"
  box TSPX, PY, TSPW, PH,, rgb(white), rgb(black)
  text TSPX+TSPW\2, PY+5, "Pouch", "CT",,, rgb(green)
  text TSPX+TSPW\2, PY+PH\2, m$, "CM", 4
end sub

' Draw the board background
sub DrawBackground col, row, x, y
  local c = WITHIN_COLOR
  if GetBoardRegion(col, row) = BEYOND then c = BEYOND_COLOR
  box x, y, CSIZE, CSIZE,, rgb(lightgray), c
end sub

' return WITHIN or BEYOND
function GetBoardRegion(col, row)
  r = WITHIN
  if col = 1 or col = BOARD_NCOLS then r = BEYOND
  if row = 1 or row = BOARD_NROWS then r = BEYOND
  GetBoardRegion = r
end function

' Draw a stone
sub DrawStone x, y, stone
  local symbol, cx, c, dc, s$, nv
  local txv(39), tyv(39)
  DecodeStone stone, symbol, cx
  nv = nvertices(symbol)
  c = colors(cx)
  dc = dark_colors(cx)
  rbox x, y, CSIZE, CSIZE, STONE_CRAD, rgb(lightgray), c
  if nv > 0 then
    math set 0, txv()
    math set 0, tyv()
    math slice xv(),, symbol, txv()
    math add txv(), x+10, txv()
    math slice yv(),, symbol, tyv()
    math add tyv(), y+9, tyv()
    polygon nv, txv(), tyv(), rgb(black), rgb(black)
  else
    circle x+CSIZE\2, y+CSIZE\2, 15,,, rgb(black), rgb(black)
  end if
end sub

' Process user input from the keyboard
sub GameLoopKeyboard
  local z$, cmd, cursor_col, cursor_row, cursor_x, cursor_y
  cursor_col = 6 : cursor_row = 5
  cursor_x = BOARD_X + (cursor_col-1)*CSIZE + CSIZE\2
  cursor_y = BOARD_Y + (cursor_row-1)*CSIZE + CSIZE\2
  gui cursor on 1, 0, 0, rgb(yellow)
  gui cursor show
  gui cursor cursor_x, cursor_y
  z$ = INKEY$
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case UP
        if cursor_row > 1 then inc cursor_row, -1
      case DOWN
        if cursor_row < BOARD_NROWS then inc cursor_row
      case LEFT
        if cursor_col > 1 then inc cursor_col, -1
      case RIGHT
        if cursor_col < BOARD_NCOLS then inc cursor_col
      case ENTER
        ShowMessage ""
        DoMove cursor_col, cursor_row
      case BKSPC
        UndoMove
      case F1
        ShowRules
      case ESC
        Quit
    end select
    cursor_x = BOARD_X + (cursor_col-1)*CSIZE + CSIZE\2
    cursor_y = BOARD_Y + (cursor_row-1)*CSIZE + CSIZE\2
    gui cursor cursor_x, cursor_y
  loop
end sub

' Process user input from the mouse and keyboard
sub GameLoopMouse
  local row, col, cmd, z$
  z$ = INKEY$
  do
    z$ = INKEY$
    if z$ <> "" then
      cmd = asc(UCASE$(z$))
      select case cmd
        case BKSPC
          UndoMove
        case F1
          ShowRules
        case F12
          RestartGame
        case HOME
          NewGame
        case ESC
          Quit
      end select
    end if
    if left_click then
      col = (mouse_X - BOARD_X)\CSIZE + 1
      row = (mouse_Y - BOARD_Y)\CSIZE + 1
      ShowMessage ""
      DoMove col, row
      left_click = 0
    end if
  loop
end sub

' Perform a move, detect end of game
sub DoMove col, row
  local stone, value, nstone
  stone = pouch(num_pouch_stones)
  if stone = 0 then exit sub
  if board(col, row) <> EMPTY then
    ShowMessage("Invalid move: board cell occupied")
    exit sub
  end if
  value = IsValidMove(col, row, stone)
  if value = INVALID then
    ShowMessage("Invalid move: cannot match a stone")
    exit sub
  end if
  DrawStoneFromPouch stone
  board(col, row) = stone
  PushMove col, row, stone, score
  UpdateScore value
  if num_pouch_stones = 0 then
    ShowMessage("Pouch Exhausted -- Game Over")
    UpdateScore pscores(1)
    DrawGame
    running = 0
    pause 2000
    ShowWin
  else
    nstone = pouch(num_pouch_stones)
    if HasValidMove(nstone) = 0 then
      ShowMessage("There are no valid moves -- Game Over")
      DrawGame
      running = 0
      Pause 2000
      ShowWin
    end if
  end if
  DrawGame
end sub

' Returns move value if valid move, INVALID if not
function IsValidMove(col, row, stone)
  local i, n, symbol, cx, ok
  local neighbors(4)
  local nsymbols(4), ncxs(4)
  if stone = 0 or board(col, row) <> EMPTY then
    IsValidMove = INVALID
    exit function
  end if
  DecodeStone stone, symbol, cx
  GetNeighbors col, row, n, neighbors()
  for i = 1 to n
    DecodeStone neighbors(i), nsymbols(i), ncxs(i)
  next i
  select case n
    case 0
      if GetBoardRegion(col, row) = BEYOND then
        IsValidMove = 0
      else
        IsValidMove = INVALID
      end if
    case 1
      if nsymbols(1) = symbol or ncxs(1) = cx then
        if GetBoardRegion(col, row) = BEYOND then
          IsValidMove = 0
        else
          inc score_parts(1)
          IsValidMove = mscores(n)
        end if
      else
        IsValidMove = INVALID
      end if
    case 2
      ok = 0
      if nsymbols(1) = symbol and ncxs(2) = cx then ok = 1
      if nsymbols(2) = symbol and ncxs(1) = cx then ok = 1
      if ok then
        if GetBoardRegion(col, row) = BEYOND then
          IsValidMove = 0
        else
          inc score_parts(2)
          IsValidMove = mscores(2)
        end if
      else
        IsValidMove = INVALID
      end if
    case 3
      ok = 0
      if nsymbols(1) = symbol and nsymbols(2) = symbol and ncxs(3) = cx then ok = 1
      if nsymbols(1) = symbol and nsymbols(3) = symbol and ncxs(2) = cx then ok = 1
      if nsymbols(2) = symbol and nsymbols(3) = symbol and ncxs(1) = cx then ok = 1
      if ncxs(1) = cx and ncxs(2) = cx and nsymbols(3) = symbol then ok = 1
      if ncxs(1) = cx and ncxs(3) = cx and nsymbols(2) = symbol then ok = 1
      if ncxs(2) = cx and ncxs(3) = cx and nsymbols(1) = symbol then ok = 1
      if ok then
        if GetBoardRegion(col, row) = BEYOND then
          IsValidMove = 0
        else
          inc score_parts(3)
          IsValidMove = mscores(3)
        end if
      else
        IsValidMove = INVALID
      end if
    case 4
      ok = 0
      if nsymbols(1) = symbol and nsymbols(2) = symbol and ncxs(3) = cx and ncxs(4) = cx then ok=1
      if nsymbols(1) = symbol and nsymbols(3) = symbol and ncxs(2) = cx and ncxs(4) = cx then ok=1
      if nsymbols(1) = symbol and nsymbols(4) = symbol and ncxs(2) = cx and ncxs(3) = cx then ok=1
      if nsymbols(2) = symbol and nsymbols(3) = symbol and ncxs(1) = cx and ncxs(4) = cx then ok=1
      if nsymbols(2) = symbol and nsymbols(4) = symbol and ncxs(1) = cx and ncxs(3) = cx then ok=1
      if nsymbols(3) = symbol and nsymbols(4) = symbol and ncxs(1) = cx and ncxs(2) = cx then ok=1
      if ok then
        if GetBoardRegion(col, row) = BEYOND then
          IsValidMove = 0
        else
          inc score_parts(4)
          IsValidMove = mscores(4) + bonuses(score_parts(4))
        end if
      else
        IsValidMove = INVALID
      end if
  end select
end function

' Return a list of the neighboring stones for a given board cell
sub GetNeighbors col, row, n, neighbors()
  local d, ncol, nrow
  n = 0
  for d = 1 to 4
    select case d
      case 1: ncol = col-1 : nrow = row
      case 2: nrow = row-1 : ncol = col
      case 3: ncol = col+1 : nrow = row
      case 4: nrow = row+1 : ncol = col
    end select
    if ncol < 1 or ncol > BOARD_WIDTH then continue for
    if nrow < 1 or nrow > BOARD_HEIGHT then continue for
    if board(ncol, nrow) = EMPTY then continue for
    inc n
    neighbors(n) = board(ncol, nrow)
  next d
end sub  

' Update the score
sub UpdateScore value
  inc score, value
end sub

' See if there are any valid moves for the next stone
function HasValidMove(stone)
  local row, col
  for row = 1 to BOARD_NROWS
    for col = 1 to BOARD_NCOLS
      if IsValidMove(col, row, stone) <> INVALID then
        HasValidMove = 1
        exit function
      end if
    next col
  next row
  HasValidMove = 0
end function

' Push a move onto the undo stack
sub PushMove col, row, stone, score
  local move
  move = col*100 + row
  inc sp
  stack(1, sp) = move
  stack(2, sp) = stone
  stack(3, sp) = score
end sub

' Undo a game move
sub UndoMove
  local move, stone, col, row
  if sp = 0 then exit sub
      move = stack(1, sp)
  stone = stack(2, sp)
  score = stack(3, sp)
  inc sp, -1
  col = move\100
  row = move - 100*col
  board(col, row) = EMPTY  
  inc num_pouch_stones
  pouch(num_pouch_stones) = stone
  DrawGame
end sub

' restart a game
sub RestartGame
  if not running then
    ShowMessage("Game is Over -- Cannot Restart")
    exit sub
  end if
  math add restart_pouch(), 0, pouch()
  math add restart_board(), 0, board()
  num_pouch_stones = NUM_STONES - 6
  DrawGame
end sub

' Show the Start Screen
sub ShowStartScreen
  local i, x, y, z$, cmd
  cls
  gui cursor hide
  text mm.hres\2, 10, "Ishido - The Way of Stones", "CT", 4,, rgb(green)
  FillPouch
  DoInitialDeal
  DrawStone 10, 100, board(iblocs(1, 1), iblocs(2, 1))
  DrawStone 10, 300, board(iblocs(1, 2), iblocs(2, 2))
  DrawStone 10, 500, board(iblocs(1, 3), iblocs(2, 3))
  DrawStone 750, 100, board(iblocs(1, 4), iblocs(2, 4))
  DrawStone 750, 300, board(iblocs(1, 5), iblocs(2, 5))
  DrawStone 750, 500, board(iblocs(1, 6), iblocs(2, 6))
  DrawTopScores
  text mm.hres\2, 550, "Press F1 for Rules or Space to Play", "CT"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cmd = asc(UCASE$(z$))
  select case cmd
    case F1
      ShowRules
    case SPACE
      exit do    
  end select
end sub

' Draw the Top Scores box
sub DrawTopScores
  local x, y, num, spos(MAX_TOP_SCORES)
  GetTopScores num, spos()
  box TOPSCORE_X, TOPSCORE_Y, TOPSCORE_W, TOPSCORE_H
  text TOPSCORE_X+TOPSCORE_W\2, TOPSCORE_Y-3, "TOP SCORES", "CB", 4
  x = TOPSCORE_X + 10
  y = TOPSCORE_Y + 10
  for i = 1 to num
    text x, y, topscore_names$(spos(i))
    text x+TOPSCORE_W\2, y, str$(topscore_values(i))
    inc y, 20
  next i
end sub

' Fetch the top scores from the file
sub GetTopScores num, spos()
  local buf$, x, y
  on error skip 1
  open "ishido_scores.txt" for input as #2
  if mm.errno <> 0 then exit sub
  num = 0
  line input #2, buf$
  do
    inc num
    cat buf$, ","
    topscore_names$(num) = field$(buf$, 1, ",")
    topscore_values(num) = val(field$(buf$, 2, ","))
    line input #2, buf$
  loop until buf$ = ""
  close #2
  sort topscore_values(), spos(), 1, 1, num
end sub

' Update the top scores list
sub UpdateTopScores name$, num, new_score
  local i
  on error skip 1
  open "ishido_scores.txt" for output as #2
  if mm.errno <> 0 then exit sub
  if num < MAX_TOP_SCORES then
    inc num
  end if
  topscore_values(num) = new_score
  topscore_names$(num) = name$
  for i = 1 to num
    m$ = topscore_names$(i) + "," + str$(topscore_values(i))
    print #2, m$
  next i
  close #2
end sub

' Solicit Initials from User for Top Score List
sub GetInitials name$
  box 300, 200, 400, 100,, rgb(white), rgb(black)
  print @(310, 210); "NEW TOP SCORE!"
  print @(310, 240); "Enter your initials (3 chars max): ";
  input "", n$
  name$ = UCASE$(LEFT$(n$, 3))
  ShowStartScreen
end sub

' Show stats for a won game
sub ShowWin
  local i, j, y, value, pb, m$, n$, v$, t$, s$
  local num, spos(MAX_TOP_SCORES)
  gui cursor hide
  box WIN_X, WIN_Y, WIN_W, WIN_H,, rgb(white), rgb(black)
  text WIN_X+WIN_W\2, WIN_Y+10, "Game Over", "CT", 5,, rgb(yellow)
  y = WIN_Y + 70
  for i = 1 to 4
    t$ = "Number of " + str$(i) + "-way matches: "
    n$ = str$(score_parts(i))
    value = score_parts(i)*mscores(i)
    if i = 4 then
      for j = 1 to score_parts(i)
        value = value + bonuses(j)
      next j
    end if
    v$ = "Value: " + str$(value)
    m$ = t$ + n$
    s$ = space$(40 - len(m$))
    cat m$, s$
    cat m$, v$
    text WIN_X+40, y, m$
    inc y, 20
  next i
  pb = 0
  if num_pouch_stones <= 2 then
    pb = pscores(num_pouch_stones+1)
  end if
  t$ = "Pouch Bonus: "
  s$ = space$(40 - len(t$))
  m$ = t$ + s$ + "Value: " + str$(pb)
  text WIN_X+40, y, m$
  t$ = "Total Score:
  m$ = t$ + space$(40- len(t$)) + "Value: " + str$(score)
  inc y, 20
  text WIN_X+40, y, m$,,,, rgb(green)
  GetTopScores num, spos()
  if (num < MAX_TOP_SCORES) or (score > topscore_values(num)) then
    inc y, 40
    text WIN_X+40, y, "You have a new Top Score!"
    inc y, 40
    text WIN_X+40, y, "Enter your initials (3 chars max): "
    input "", name$
    UpdateTopScores UCASE$(LEFT$(name$,4)), num, score
  end if
  text WIN_X+WIN_W\2, WIN_Y+WIN_H-3, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  DrawGame
end sub

' Show the rules
sub ShowRules
  local z$
  cls
  text mm.hres\2, 10, "Ishido Rules", "CT", 4,, rgb(green)
  text 0, 30, ""
  print "Ishido is a game of symbol and color matching. There are 6 different symbols and"
  print "6 different colors. Each possible combination of symbol and color is present in  a"
  print "'stone', and each unique stone is repeated, giving a total of 6 times 6 times 2,"
  print "or 72 stones in all. At the start of a game, 6 stones have already been placed on"
  print "the board. They show the 6 symbols and 6 colors with no duplicates of either."
  print ""
  print "The 'touchstone' at the top right shows the next stone to be placed. You must decide"
  print "where to place it on the board. For the square you want to place the stone, its"
  print "immediate neighbors to top, bottom, left, and right (but not diagonally) must match"
  print "at least one attribute of the stone you want to place. That is, each neighbor must"
  print "have either the same symbol or the same color, or both, as the new stone."
  print ""
  print "The board cells have a darker background color on the edges of the board. These cells"
  print "are the BEYOND. The board cells inside the BEYOND are the WITHIN. There are different"
  print "rules for placing cells in each region: In the BEYOND, you can place stones that have"
  print "no neighbors at all, but you get no score for any stone in BEYOND. In the WITHIN, your"
  print "new stones must have at least one matching neighbor, and you get score values for these"
  print "placements."
  print ""
  print "If your new stone matches a single neighboring stone, either by symbol or by color, you"
  print "get 1 point. If the new stone matches one neighbor by symbol and another neighbor by"
  print "color, you get 2 points, but you can't get 2 points with a single neighbor that happens"
  print "to match both symbol and color: only 1 point is awarded. If your stone matches two"
  print "distinct neighbors with either symbol or color, and one other neighbor with the other"
  print "attribute, you get 4 points. That is, you need 3 neighboring stones, and match 2 of them"
  print "by symbol and the third by color, or two by color and the third by symbol. If you have"
  print "4 neighboring stones, and you match 2 by symbol and the other 2 by color, then you get"
  print "8 points and a large bonus. The bonus depends on how many total 4-way matches you have."
  print "The schedule is 25, 50, 100, 200, 400, 600, 800, 1000, 5000, 10000, 25000, 50000 for 1,"
  print "2, 3, 4, and so on 4-way matches."
  print ""
  print "You also get bonus points if you manage to place all or nearly all of the 72 stones onto"
  print "the board. If the pouch ends up with 2 stones after you run out of moves, you get a 100"
  print "point bonus; if it has only 1 stone, you get a 500 point bonus, and if there are no stones"
  print "the bonus is 1000 points. The 'pouch' box at lower right shows you the number of stones"
  print "remaining in the pouch."
  print ""
  print "The game will tell you when you have no remaining moves."
  text mm.hres\2, 540, "Press Any Key to Continue", "CT"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  ShowRules2
end sub

' More rules
sub ShowRules2
  cls
  text mm.hres\2, 10, "How to Play", "CT", 4,, rgb(green)
  text 0, 30, ""
  print "You can play Ishido either using the keyboard only or with a mouse."
  print ""
  print "To play Ishido, using the keyboard only, Use the arrow keys to move the cross cursor up,"
  print "down, left, and right to navigate to the board cell into which you want the to place the"
  print "stone displayed in the touchstone window. If the move is legal, the stone will appear in"
  print "that board cell. If the move is not legal, an error message will appear at the bottom of"
  print "the screen."
  print ""
  print "Play using a mouse is pretty much the same, except that you click the board cell using the"
  print "left mouse button."
  print ""
  print "Here are the additional keyboard commands that work both with and without a mouse:"
  print "   F1        -- Shows these help screens again"
  print "   F12       -- Restarts your current game"
  print "   Home      -- Abandons the current game and starts a new game"
  print "   Backspace -- Undo moves"
  print "   Escape    -- Quit the program"
  print ""
  print "An Ishido game terminates when one of two things happens:"
  print "  1. All the stones in the pouch have been played onto the board."
  print "  2. There are remaining stones in the pouch but no remaining legal moves for the stone"
  print "      that is shown in the touchstone window."
  print "When the game is over, a new window opens that shows you how the final game score is arrived"
  print "at by adding up all the score components and bonuses.
  print ""
  print "Should your score be higher than the current top 10 scores list, it will be added to that"
  print "list."  
  text mm.hres\2, 540, "Press Any Key to Continue", "CT"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  DrawGame
end sub

' Show a message
sub ShowMessage m$
  local x, y
  x = BOARD_X + BOARD_WIDTH\2
  y = 550
  text x, y, space$(80), "CT"
  text x, y, m$, "CT"
end sub

' Quit the Game
sub Quit
  if hasMouse then
    settick 0,0
    controller mouse close
  end if
  gui cursor hide
  cls
  end
end sub

' return a uniformly distributed random integer in the specified closed range
function RandomIntegerInRange(a, b)
  local v, c
  c = b-a+1
  do
    v = a + (b-a+2)*rnd()
    if v >= a and v <= b then exit do
  loop
  RandomIntegerInRange = v
end function

' initial deal board coords
data 6, 4, 7, 5, 1, 1, 12, 1, 1, 8, 12, 8

' bonus values for 4-way moves
data 25, 50, 100, 200, 400, 600, 800, 1000, 5000, 10000, 25000, 50000

' polygon vertices for symbols

' Heart
data  15,6, 11,3, 10,3,  9,2,  7,2,   6,3,   5,3,   3,5,   3,6,   2,7,   2,9,  1,10,  1,13, 2,14
data  2,16, 3,17, 3,18, 4,19, 4,20, 15,30, 26,20, 26,19, 27,18, 27,17, 28,16, 28,14, 29,13, 30,10
data  29,9, 29,7, 28,6, 28,5, 28,3,  27,3,  26,2,  24,2,  23,2,  22,3,  15,6

' Star
data 15,1, 12,12, 1,12, 9,19, 6,30, 15,24, 25,30, 21,19, 30,12, 18,12, 15,1

' Square
data 5,5, 5,25, 25,25, 25,5, 5,5

' Cross
data 12,1, 12,12, 1,12, 1,19, 12,19, 12,30, 19,30, 19,19, 30,19, 30,12, 19,12, 19,1, 12,1

' Triangle
data 15,1, 1,30, 30,30, 15, 1

' Circle (no vertices needed)
